#!/bin/bash
# -*- mode: shell-script; indent-tabs-mode: nil; sh-basic-offset: 4; -*-
# ex: ts=8 sw=4 sts=4 et filetype=sh

ROOT="$1"

if [[ ! -d "$ROOT" ]]; then
    echo "Usage: $0 <rootdir>"
    exit 1
fi

if [[ "$ROOT" -ef / ]]; then
    echo "Can't convert the running system."
    echo "Please boot with 'rd.convertfs' on the kernel command line,"
    echo "to update with the help of the initramfs,"
    echo "or run this script from a rescue system."
    exit 1
fi

while [[ "$ROOT" != "${ROOT%/}" ]]; do
    ROOT=${ROOT%/}
done

if [ ! -L $ROOT/var/run -a -e $ROOT/var/run ]; then
    echo "Converting /var/run to symlink"
    mv -f $ROOT/var/run $ROOT/var/run.runmove~
    ln -sfn ../run $ROOT/var/run
    # Prevent the clean script from running. It's not needed now but if it's still
    # active on boot, it will remove the symlinks we've just created above!
    rm -f "$ROOT/lib/systemd/system/sysinit.target.wants/mandriva-clean-var-run-lock.service"
fi

if [ ! -L $ROOT/var/lock -a -e $ROOT/var/lock ]; then
    echo "Converting /var/lock to symlink"
    mv -f $ROOT/var/lock $ROOT/var/lock.lockmove~
    ln -sfn ../run/lock $ROOT/var/lock
    # Prevent the clean script from running. It's not needed now but if it's still
    # active on boot, it will remove the symlinks we've just created above!
    rm -f "$ROOT/lib/systemd/system/sysinit.target.wants/mandriva-clean-var-run-lock.service"
fi

needconvert() {
    for dir in "$ROOT/bin" "$ROOT/sbin" "$ROOT/lib" "$ROOT/lib64"; do
        if [[ -e "$dir" ]]; then
            [[ -L "$dir" ]] || return 0
        fi
    done
    return 1
}

if ! [ -e "$ROOT/usr/bin" ]; then
    echo "$ROOT/usr/bin does not exist!"
    echo "Make sure, the kernel command line has enough information"
    echo "to mount /usr (man dracut.cmdline)"
    exit 1
fi


if ! needconvert; then
    echo "Your system is already converted."
    if [[ -f "$ROOT/var/lib/mageia-prepare-upgrade/state" ]]; then
       echo "ready" >"$ROOT/var/lib/mageia-prepare-upgrade/state"
    fi
    exit 0
fi

testfile="$ROOT/.usrmovecheck$$"
rm -f -- "$testfile"
> "$testfile"
if [[ ! -e "$testfile" ]]; then
    echo "Cannot write to $ROOT/"
    exit 1
fi
rm -f -- "$testfile"

testfile="$ROOT/usr/.usrmovecheck$$"
rm -f -- "$testfile"
> "$testfile"
if [[ ! -e "$testfile" ]]; then
    echo "Cannot write to $ROOT/usr/"
    exit 1
fi
rm -f -- "$testfile"

find_mount() {
    local dev mnt etc wanted_dev
    wanted_dev="$(readlink -e -q $1)"
    while read dev mnt etc; do
        [ "$dev" = "$wanted_dev" ] && echo "$dev" && return 0
    done < /proc/mounts
    return 1
}

# usage: ismounted <mountpoint>
# usage: ismounted /dev/<device>
if command -v findmnt >/dev/null; then
    ismounted() {
        findmnt "$1" > /dev/null 2>&1
    }
else
    ismounted() {
        if [ -b "$1" ]; then
            find_mount "$1" > /dev/null && return 0
            return 1
        fi

        while read a m a; do
            [ "$m" = "$1" ] && return 0
        done < /proc/mounts
        return 1
    }
fi

# clean up after ourselves no matter how we die.
cleanup() {
    echo "Something failed. Move back to the original state"
    for dir in "$ROOT/bin" "$ROOT/sbin" "$ROOT/lib" "$ROOT/lib64" \
	"$ROOT/usr/bin" "$ROOT/usr/sbin" "$ROOT/usr/lib" \
        "$ROOT/usr/lib64"; do
        [[ -d "${dir}.usrmove-new" ]] && rm -fr -- "${dir}.usrmove-new"
        if [[ -d "${dir}.usrmove-old" ]]; then
            mv "$dir" "${dir}.del~"
            mv "${dir}.usrmove-old" "$dir"
            rm -fr -- "${dir}.del~"
        fi
    done
}

trap 'ret=$?; [[ $ret -ne 0 ]] && cleanup;exit $ret;' EXIT
trap 'exit 1;' SIGINT

ismounted "$ROOT/usr" || CP_HARDLINK="-l"

set -e

# merge / and /usr in new dir in /usr
for dir in bin sbin lib lib64; do
    rm -rf -- "$ROOT/usr/${dir}.usrmove-new"
    [[ -L "$ROOT/$dir" ]] && continue
    [[ -d "$ROOT/$dir" ]] || continue
    echo "Make a copy of \`$ROOT/usr/$dir'."
    [[ -d "$ROOT/usr/$dir" ]] \
        && cp -ax -l "$ROOT/usr/$dir" "$ROOT/usr/${dir}.usrmove-new"

    # (cg) Mageia packaging quirks....
    if [[ "$dir" = "bin" ]]; then
        # shadow-utils on mga2 packaged this binary in /usr/bin when it shouldn't...
        [[ -f "$ROOT/usr/${dir}.usrmove-new/login" ]] \
            && [[ -f "$ROOT/${dir}/login" ]] \
            && mv "$ROOT/usr/${dir}.usrmove-new/login" "$ROOT/usr/${dir}.usrmove-new/login.usrmove~"
    elif [[ "$dir" = "lib" ]]; then
        # libgcc1 on x86_64 installs had duplicate symlinks that ultimately point at the same file
        [[ -L "$ROOT/${dir}/libgcc_s.so.1" ]] \
            && [[ -L "$ROOT/usr/${dir}.usrmove-new/libgcc_s.so.1" ]] \
            && rm -f "$ROOT/usr/${dir}.usrmove-new/libgcc_s.so.1"
    fi

    echo "Merge the copy with \`$ROOT/$dir'."
    [[ -d "$ROOT/usr/${dir}.usrmove-new" ]] \
        || mkdir -p "$ROOT/usr/${dir}.usrmove-new"
    cp -axT $CP_HARDLINK --backup --suffix=.usrmove~ "$ROOT/$dir" "$ROOT/usr/${dir}.usrmove-new"
    echo "Clean up duplicates in \`$ROOT/usr/$dir'."
    # delete all symlinks that have been backed up
    find "$ROOT/usr/${dir}.usrmove-new" -type l -name '*.usrmove~' -delete || :
    # replace symlink with backed up binary
    find "$ROOT/usr/${dir}.usrmove-new" \
        -name '*.usrmove~' \
        -type f \
        -exec bash -c 'p="{}";o=${p%%%%.usrmove~};
                       [[ -L "$o" ]] && mv -f "$p" "$o"' ';' || :
done
# switch over merged dirs in /usr
for dir in bin sbin lib lib64; do
    [[ -d "$ROOT/usr/${dir}.usrmove-new" ]] || continue
    echo "Switch to new \`$ROOT/usr/$dir'."
    rm -fr -- "$ROOT/usr/${dir}.usrmove-old"
    mv "$ROOT/usr/$dir" "$ROOT/usr/${dir}.usrmove-old"
    mv "$ROOT/usr/${dir}.usrmove-new" "$ROOT/usr/$dir"
done

# replace dirs in / with links to /usr
for dir in bin sbin lib lib64; do
    [[ -L "$ROOT/$dir" ]] && continue
    [[ -d "$ROOT/$dir" ]] || continue
    echo "Create \`$ROOT/$dir' symlink."
    rm -fr -- "$ROOT/${dir}.usrmove-old" || :
    mv "$ROOT/$dir" "$ROOT/${dir}.usrmove-old"
    ln -sfn usr/$dir "$ROOT/$dir"
done

echo "Clean up backup files."
# everything seems to work; cleanup
for dir in bin sbin lib lib64; do
     # if we get killed in the middle of "rm -rf", ensure not to leave
     # an incomplete directory, which is moved back by cleanup()
    [[ -d "$ROOT/usr/${dir}.usrmove-old" ]] \
        && mv "$ROOT/usr/${dir}.usrmove-old" "$ROOT/usr/${dir}.usrmove-old~"
    [[ -d "$ROOT/${dir}.usrmove-old" ]] \
        && mv "$ROOT/${dir}.usrmove-old" "$ROOT/${dir}.usrmove-old~"
done

for dir in bin sbin lib lib64; do
    [[ -d "$ROOT/usr/${dir}.usrmove-old~" ]] \
        && rm -rf -- "$ROOT/usr/${dir}.usrmove-old~" || :
    [[ -d "$ROOT/${dir}.usrmove-old~" ]] \
        && rm -rf -- "$ROOT/${dir}.usrmove-old~" || :
done

for dir in lib lib64; do
    [[ -d "$ROOT/$dir" ]] || continue
    for lib in "$ROOT"/usr/${dir}/lib*.so*.usrmove~; do
        [[ -f $lib ]] || continue
        mv $lib ${lib/.so/_so}
    done
done

set +e

echo "Run ldconfig."
ldconfig -r "$ROOT"

[[ -f "$ROOT/etc/selinux/config" ]] && . "$ROOT/etc/selinux/config"
if [ -n "$(command -v setfiles)" ] && [ "$SELINUX" != "disabled" ] && [ -f /etc/selinux/${SELINUXTYPE}/contexts/files/file_contexts ]; then
    echo "Fixing SELinux labels"
    setfiles -r $ROOT -p /etc/selinux/${SELINUXTYPE}/contexts/files/file_contexts $ROOT/sbin $ROOT/bin $ROOT/lib $ROOT/lib64 $ROOT/usr/lib $ROOT/usr/lib64 $ROOT/etc/ld.so.cache $ROOT/var/cache/ldconfig || :
fi

# (cg) iptables package has an unpackaged symlink that conflicts with a script name
if [[ -f "$ROOT/usr/lib64/iptables.usrmove~" ]]; then
  mv "$ROOT/usr/lib64/iptables.usrmove~" "$ROOT/usr/lib64/iptables.init"
  [[ -f "$ROOT/usr/lib/systemd/system/iptables.service" ]] \
    && sed -i 's,lib64/iptables ,lib64/iptables.init ,' "$ROOT/usr/lib/systemd/system/iptables.service"
  if [[ -f "$ROOT/usr/lib64/ip6tables" ]]; then
    mv "$ROOT/usr/lib64/ip6tables" "$ROOT/usr/lib64/ip6tables.init"
    [[ -f "$ROOT/usr/lib/systemd/system/ip6tables.service" ]] \
      && sed -i 's,lib64/ip6tables ,lib64/ip6tables.init ,' "$ROOT/usr/lib/systemd/system/ip6tables.service"
  fi
elif [[ -f "$ROOT/usr/lib/iptables.usrmove~" ]]; then
  mv "$ROOT/usr/lib/iptables.usrmove~" "$ROOT/usr/lib/iptables.init"
  [[ -f "$ROOT/usr/lib/systemd/system/iptables.service" ]] \
    && sed -i 's,lib/iptables ,lib/iptables.init ,' "$ROOT/usr/lib/systemd/system/iptables.service"
  if [[ -f "$ROOT/usr/lib/ip6tables" ]]; then
    mv "$ROOT/usr/lib/ip6tables" "$ROOT/usr/lib/ip6tables.init"
    [[ -f "$ROOT/usr/lib/systemd/system/ip6tables.service" ]] \
      && sed -i 's,lib/ip6tables ,lib/ip6tables.init ,' "$ROOT/usr/lib/systemd/system/ip6tables.service"
  fi
fi

if [[ -f "$ROOT/var/lib/mageia-prepare-upgrade/state" ]]; then
    echo "ready" >"$ROOT/var/lib/mageia-prepare-upgrade/state"
fi

echo "Done."
exit 0
